/*  Gedacht zur Ausführung NACH dem zeitlichen Verschieben von AGs auf einer KSVBA-Ressource. Diese Funktion erstellt die
    Blocktimes erneut für die sich in der temporären Tabelle gemerkten AGs.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__blocktime__recreate__by__resource_id', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__blocktime__recreate__by__resource_id(
      _resource_id  integer,   -- KSVBA-Ressource auf der AGs zeitlich verschoben werden.
      _loglevel     integer DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS void AS $$

  DECLARE

    _prefix         varchar := format( 'resource_timeline__blocktime__recreate__by__resource_id resource_id:%L -', _resource_id );
    _current_time   timestamp := coalesce( nullif( TSystem.Settings__Get( 'CI.AssumedNow' ), '' )::timestamp, localtimestamp( 0 ) );   -- Aktueller Zeitpunkt
    _count          integer;
    _totalcount     integer;

    _ab2 ab2;

    -- Exception-Handling
    _errmsg text;
    _message_text text;
    _pg_exception_context text;
    _pg_exception_detail text;
    _pg_exception_hint text;

    -- Messung Dauer für die Schleifendurchläufe
    _loop_time_begin    timestamp;
    _loop_time_end      timestamp;
    _loop_time_current  interval;
    _loop_time_min      interval;
    _loop_time_max      interval;
    _loop_time_avg      interval;
    _loop_time_total    interval;


  BEGIN

      IF _loglevel >= 4 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__blocktime__recreate__by__resource_id( _resource_id => %L,  _loglevel => %L )',
                                                                                                                 _resource_id      ,  _loglevel
                                  )
          ;
      END IF;

            -- Debug.
      IF _loglevel >= 5 THEN
          RAISE NOTICE '% now:%;', _prefix, _current_time;
      END IF;

      -- Prüfen, ob temporäre Tabelle  vorhanden ist.
      IF NOT EXISTS ( SELECT true FROM information_schema.tables WHERE table_name = '_tmp_a2_ids' AND table_type = 'LOCAL TEMPORARY' ) THEN
          RAISE EXCEPTION 'Temporaray table "_tmp_a2_ids" does not exist!';
      END IF;

      -- Für die gemerkten AGs und für AGs mit DLZ-Marker die Blockierungen neu erstellen.
      _count := 0;
      _totalcount :=  count( ab2.* )
                          FROM ab2
                          JOIN abk ON ab_ix = a2_ab_ix
                         WHERE      -- Fall bezogene Filterbedingungen
                                    (
                                          -- Blockierungen wegen fehlenden Stempelungen
                                          (
                                                  a2_at < _current_time                                                                                                  -- AG beginnt in der Vergangenheit, ...
                                              AND coalesce( a2_ta, 0 ) > 0                                                                                               -- ... hat eine Laufzeit, ...
                                              -- AND coalesce( a2_dlz, 0 ) = 0                                                                                              -- ... aber keine Durchlaufzeit.
                                              -- AND (
                                              --             coalesce( ab2.modified_date, ab2.insert_date ) BETWEEN _current_time - interval '1 day' AND _current_time      -- Der AG wurde am vorhergehenden Tag angefasst ...
                                              --         OR  a2_et > _current_time - interval '1 day'                                                                       -- ... oder hat kürzlich geendet oder endet bald.
                                              --     )
                                              AND EXISTS( SELECT true FROM _tmp_a2_ids AS i1 WHERE i1.ta_a2_id = ab2.a2_id AND i1.ta_resource_id = _resource_id )        -- AG ist in Liste der gemerkten AGs für gegebene Ressource.
                                          )
                                          -- Blockierungen wegen Überlappungen
                                      OR  (
                                              EXISTS( SELECT true FROM scheduling.resource_timeline AS i2 WHERE i2.ti_a2_id = ab2.a2_id AND i2.ti_stat = 'dlz' AND i2.ti_resource_id = _resource_id )  -- Der AG wurde mit DLZ auf der gegebenen Ressource einterminiert ...
                                              -- AND coalesce( ab2.modified_date, ab2.insert_date ) BETWEEN _current_time - interval '1 day' AND _current_time                                -- ... und wurde im letzten Tag angefasst.
                                          )
                                    )
                                    -- allgemeine Filterbedingungen
                                AND a2_interm                                                                                     -- AG ist in Plantafel
                                AND EXISTS( SELECT true FROM scheduling.resource_timeline AS i3 WHERE i3.ti_a2_id = ab2.a2_id )  -- AG ist wirklich einterminiert
                                AND NOT a2_ende                                                                                   -- AG ist noch offen
                                AND NOT ab_done                                                                                   --   -"-
                                AND NOT a2_buch                                                                                   -- AG ist noch nicht nachkalkuliert
                                AND NOT a2_ausw                                                                                   -- AG ist auf inhaus Kostenstellen
                                AND NOT a2_sperr                                                                                  -- AG ist nicht gesperrt
                      ;

      FOR _ab2 IN
          SELECT ab2.*
            FROM ab2
            JOIN abk ON ab_ix = a2_ab_ix
           WHERE       -- Fall bezogene Filterbedingungen
                      (
                            -- Blockierungen wegen fehlenden Stempelungen
                            (
                                    a2_at < _current_time    -- AG beginnt in der Vergangenheit, ...
                                AND coalesce( a2_ta, 0 ) > 0    -- ... hat eine Laufzeit, ...
                                -- AND coalesce( a2_dlz, 0 ) = 0   -- ... aber keine Durchlaufzeit.
                                -- AND (
                                --             coalesce( ab2.modified_date, ab2.insert_date ) BETWEEN _current_time - interval '1 day' AND _current_time -- Der AG wurde am vorhergehenden Tag angefasst ...
                                --        OR  a2_et > _current_time - interval '1 day'                                                                     -- ... oder hat kürzlich geendet oder endet bald.
                                --     )
                                AND EXISTS( SELECT true FROM _tmp_a2_ids AS i1 WHERE i1.ta_a2_id = ab2.a2_id AND i1.ta_resource_id = _resource_id )          -- AG ist in Liste der gemerkten AGs für gegebene Ressource.
                            )
                            -- Blockierungen wegen Überlappungen
                        OR  (
                                EXISTS( SELECT true FROM scheduling.resource_timeline AS i2 WHERE i2.ti_a2_id = ab2.a2_id AND i2.ti_stat = 'dlz' AND i2.ti_resource_id = _resource_id )  -- Der AG wurde mit DLZ auf der gegebenen Ressource einterminiert ...
                                -- AND coalesce( ab2.modified_date, ab2.insert_date ) BETWEEN _current_time - interval '1 day' AND _current_time                                -- ... und wurde im letzten Tag angefasst.
                            )
                      )
                      -- allgemeine Filterbedingungen
                  AND a2_interm                                                                                     -- AG ist in Plantafel
                  AND EXISTS( SELECT true FROM scheduling.resource_timeline AS i3 WHERE i3.ti_a2_id = ab2.a2_id )   -- AG ist wirklich einterminiert
                  AND NOT a2_ende                                                                                   -- AG ist noch offen
                  AND NOT ab_done                                                                                   --   -"-
                  AND NOT a2_buch                                                                                   -- AG ist noch nicht nachkalkuliert
                  AND NOT a2_ausw                                                                                   -- AG ist auf inhaus Kostenstellen
                  AND NOT a2_sperr                                                                                  -- AG ist nicht gesperrt
          ORDER BY a2_at, a2_et

      LOOP
          _count := _count + 1;
          _loop_time_begin := clock_timestamp();

          -- Debug
          IF _loglevel >= 4 THEN
              RAISE NOTICE '% [% / %] a2_id:%, a2_ab_ix:%, a2_n:%;', _prefix, _count, _totalcount, _ab2.a2_id, _ab2.a2_ab_ix, _ab2.a2_n;
          END IF;

          -- Exception-Handling-Block um eventuelle Fehler bei der Terminierung des aktuellen AG abzufangen und danach weiterarbeiten zu können.
          BEGIN
              PERFORM scheduling.resource_timeline__ab2__retermination_fragmented( _ab2 );

          EXCEPTION
              WHEN OTHERS THEN
                  GET STACKED DIAGNOSTICS _message_text = MESSAGE_TEXT
                                        , _pg_exception_detail = PG_EXCEPTION_DETAIL
                                        , _pg_exception_hint = PG_EXCEPTION_HINT
                                        , _pg_exception_context = PG_EXCEPTION_CONTEXT;
                  _errmsg :=  'ERROR: ' || _message_text || E'\n' ||
                              'Detail:' || _pg_exception_detail || E'\n' ||
                              'Context:' || _pg_exception_context || E'\n' ||
                              'Hint:' || _pg_exception_hint || E'\n';
                  RAISE WARNING '%', _errmsg;
          END;

          -- Messung Dauer für die Schleifendurchläufe
          _loop_time_end := clock_timestamp();
          _loop_time_current := ( _loop_time_end - _loop_time_begin );
          -- Min
          IF _loop_time_min IS null OR _loop_time_min > _loop_time_current  THEN
              _loop_time_min := _loop_time_current;
          END IF;
          -- Max
          IF _loop_time_max IS null OR _loop_time_max < _loop_time_current THEN
              _loop_time_max := _loop_time_current;
          END IF;
          -- Total
          IF _loop_time_total IS null THEN
              _loop_time_total := _loop_time_current;
          ELSE
              _loop_time_total := _loop_time_total + _loop_time_current;
          END IF;
          -- Debug: Duration
          IF _loglevel >= 5 THEN
              RAISE NOTICE '% duration of [ab2: %] : %',
                  _prefix,
                  _ab2.a2_id, _loop_time_current;
              -- Debug: Duration Statistics
              RAISE NOTICE '% duration statistics : min:%, avg:%, max:%;',
                  _prefix,
                  _loop_time_min, ( _loop_time_total / _count ), _loop_time_max;
          END IF;

      END LOOP;

      -- Temporäre Tabelle wieder aufräumen
      DELETE FROM _tmp_a2_ids
      WHERE ta_resource_id = _resource_id;

  END $$ LANGUAGE plpgsql;